<?php

if (!defined('ABSPATH')) {
    exit; // Exit if accessed directly
}

if (!class_exists('WC_Email_Customer_RFQ_Sent')) :

    /**
     * Customer Processing RFQ Email
     *
     * An email sent to the customer when a new RFQ is received.
     * @extends     WC_Email
     */
    class WC_Email_Customer_RFQ_Sent extends WC_Email
    {

        /**
         * Constructor
         */
        function __construct()
        {

            $this->id = 'customer_rfq_sent';
            $this->title = __('RFQ-ToolKit Quote Sent', 'rfqtk');
            $this->description = __('This is the quote sent to customers containing their quote detait. 
            The content of this email can be enhanced by using the proposal section in the order.', 'rfqtk');

            $this->heading = __('Your Quote for <br  /> Request #({order_number})', 'rfqtk');

            $this->subject = __('Your Quote for Request #({order_number}) - {order_date}', 'rfqtk');

            $this->content_intro = $this->format_string($this->get_option( 'content_intro' ));

            $this->template_html = 'emails/customer-rfq-sent.php';
            $this->template_plain = 'emails/plain/customer-rfq-sent.php';
            $this->_templates = array($this->template_html, $this->template_plain);


            // Triggers for this email

            $this->customer_email = true;
            $this->recipient = $this->get_option( 'recipient', '' );

            add_filter('woocommerce_template_directory', array($this, 'gpls_rfq_woocommerce_locate_template_dir'), 10, 2);
            add_action('woocommerce_order_status_gplsquote-sent_notification', array($this, 'trigger'));




            // Call parent constructor
            parent::__construct();


        }

        public function rfqtk_email_proposal_function($order, $sent_to_admin, $plain_text)
        {

             wc_get_template('woo-rfq/rfqtk-proposal.php',
                array('order' => $order,'sent_to_admin'=>$sent_to_admin,'plain_text'=>$plain_text));

        }


        public function gpls_rfq_woocommerce_locate_template_dir($dir, $template)
        {
            return $dir;
        }

        /**
         * Trigger.
         */
        function trigger($order_id)
        {
            if(!has_action('rfqtk_email_proposal',array($this, 'rfqtk_email_proposal_function'))) {
                add_action('rfqtk_email_proposal', array($this, 'rfqtk_email_proposal_function'), 100, 3);
            }
            if ($order_id) {
                $order = WC_Order_Factory::get_order($order_id);
                $order->update_meta_data('_gpls_woo_is_quote','yes');
                $order->save();
            }

            if ($order_id) {
                $this->object = wc_get_order($order_id);
                $this->recipient = $this->get_option( 'recipient').','.$this->object->get_billing_email();

                $this->find['order-date'] = '{order_date}';
                $this->find['order-number'] = '{order_number}';

                $this->replace['order-date'] = date_i18n(wc_date_format(), strtotime($this->object->get_date_created()));
                $this->replace['order-number'] = $this->object->get_order_number();
            }

            if (!$this->is_enabled() || !$this->get_recipient()) {
                return;
            }

            $this->send($this->get_recipient(), $this->get_subject(), $this->get_content(), $this->get_headers(), $this->get_attachments());

        }


        /**
         * get_content_html function.
         *
         * @access public
         * @return string
         */
        function get_content_html()
        {


            return wc_get_template_html($this->template_html, array(
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'content_intro' => $this->content_intro,
                'sent_to_admin' => false,
                'plain_text' => false,
                'email'			=> $this,
            ));

        }

        /**
         * get_content_plain function.
         *
         * @access public
         * @return string
         */
        function get_content_plain()
        {
            return wc_get_template_html($this->template_plain, array(
                'order' => $this->object,
                'email_heading' => $this->get_heading(),
                'content_intro' => $this->content_intro,
                'sent_to_admin' => false,
                'plain_text' => true,
                'email'			=> $this,

            ));

        }

        /**
         * Initialise settings form fields
         */
        public function init_form_fields()
        {
            $this->form_fields = array(
                'enabled' => array(
                    'title' => __('Enable/Disable', 'rfqtk'),
                    'type' => 'checkbox',
                    'label' => __('Enable this email notification', 'rfqtk'),
                    'default' => 'yes'
                ),
                'recipient'  => array(
                    'title'       => __( 'Additional Recipient(s)', 'rfqtk' ),
                    'type'        => 'text',
                    /* translators: %s: WP admin email */
                    'description' => sprintf( __( 'Enter recipients ( in addition to the customer) comma separated for this email.', 'rfqtk' ), '<code>' . esc_attr( get_option( 'admin_email' ) ) . '</code>' ),
                    'placeholder' => '',
                    'default'     => '',
                    'desc_tip'    => true,
                ),

                'subject' => array(
                    'title' => __('Subject', 'rfqtk'),
                    'type' => 'text',
                    'description' => sprintf(__('This controls the email subject line. Leave blank to use the default subject: <code>%s</code>.', 'rfqtk'), $this->subject),
                    'placeholder' => '',
                    'default' => ''
                ),
                'heading' => array(
                    'title' => __('Email Heading', 'rfqtk'),
                    'type' => 'text',
                    'description' => sprintf(__('This controls the main heading contained within the email notification. Leave blank to use the default heading: <code>%s</code>.', 'rfqtk'), $this->heading),
                    'placeholder' => '',
                    'default' => ''
                ),
                'content_intro' => array(
                    'title' => __('Content Intro', 'rfqtk'),
                    'type' => 'textarea',
                    'description' => sprintf(__('This controls the first paragrah contained within the email notification.', 'rfqtk'), $this->heading),
                    'placeholder' => '',
                    'default' => 'Thanks for your inquiry',

                ),
                'email_type' => array(
                    'title' => __('Email type', 'rfqtk'),
                    'type' => 'select',
                    'description' => __('Choose which format of email to send.', 'rfqtk'),
                    'default' => 'html',
                    'class' => 'email_type wc-enhanced-select',
                    'options' => $this->get_email_type_options()
                )
            );
        }


    }

endif;

return new WC_Email_Customer_RFQ_Sent();
