<?php

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly
}

if ( ! class_exists( 'WCPBC_License_Settings' ) && class_exists( 'WC_Settings_API' ) ) :

/**
 *
 * @class WCPBC_License_Settings
 * @version	2.1.10
 */
class WCPBC_License_Settings extends WC_Settings_API {

	/**
	 * The single instance of the class.
	 *
	 * @var WCPBC_License_Settings
	 */
	protected static $_instance = null;

	/**
	 * @var string
	 */
	protected $status = '';

	/**
	 * @var string
	 */
	protected $license_key = '';

	/**
	 * @var string
	 */
	protected $api_key = '';

	/**
	 * Constructor.
	 */
	public function __construct() {

		// The plugin ID. Used for option names
		$this->plugin_id = 'wc_price_based_country_';

		// ID of the class extending the settings API. Used in option names.
		$this->id = 'license';

		// Init form fields.
		$this->init_form_fields();

		// Define user set variables.
		$this->license_key  = $this->get_option( 'license_key' );
		$this->api_key  	= $this->get_option( 'api_key' );
	}

	/**
	 * Get transient status key
	 *
	 * @return string
	 */
	protected function get_transient_status_key() {
		return $this->plugin_id . $this->id . '_status';
	}

	/**
	 * Get license status
	 *
	 * @return string
	 */
	protected function get_license_status() {

		$tkey = $this->get_transient_status_key();

		if ( empty( $this->api_key ) || empty ( $this->license_key ) ) {
			$status = 'no-active';

		} elseif ( false == ( $status = get_transient( $tkey ) ) ) {

			$status = 'active';

			$response = WC_Plugin_API_Wrapper::status_check( $this->license_key, $this->api_key );

			if ( is_wp_error( $response ) ) {
				$status = 'no-active';
			}
			set_transient( $tkey, $status, 2 * HOUR_IN_SECONDS );

		}

		return $status;
	}

	/**
	 * Toggle status to active
	 */
	protected function set_status_active(){
		$this->status = 'active';
	}

	/**
	 * Toggle status to no-active
	 */
	protected function set_status_no_active(){
		$this->status = 'no-active';
	}

	/**
	 * License key getter
	 * @return string
	 */
	public function get_license_key() {
		return $this->license_key;
	}

	/**
	 * API key getter
	 * @return string
	 */
	public function get_api_key() {
		return $this->api_key;
	}

	/**
	 * Status Getter
	 */
	public function get_status() {
		if ( empty( $this->status ) ) {
			$this->status = $this->get_license_status();
		}
		return $this->status;
	}

	/**
	 * Is license active
	 * @return boolean
	 */
	public function is_status_active() {
		return ( 'active' === $this->get_status() );
	}

	/**
	 * Initialize settings form fields.
	 */
	public function init_form_fields() {

		$this->form_fields = array(
			'status' => array(
				'title'             => __( 'License status', 'wc-price-based-country-pro' ),
				'type'              => 'status_info'
			),
			'toggle' => array(
				'title'             => __( 'Toggle activation', 'wc-price-based-country-pro' ),
				'type'              => 'toggle_activation'
			),
			'license_key' => array(
				'title'             => __( 'License Key', 'wc-price-based-country-pro' ),
				'type'              => 'license_key',
				'description'       => __( 'Enter your Price Based on Country Pro license key', 'wc-price-based-country-pro' ),
				'desc_tip'          => true,
				'default'           => '',
				'custom_attributes' => array( 'autocomplete' => 'off' )
			)
		);
	}

	/**
	 * Output the admin options table.
	 */
	public function admin_options() {

		if ( $this->is_status_active() ) {
			$GLOBALS['hide_save_button'] = true;
		}
		$this->display_errors();
		parent::admin_options();
	}

	/**
	 * Processes and saves options.
	 *
	 * @return bool
	 */
	public function process_admin_options() {
		$this->init_settings();

		$fields 	= $this->get_form_fields();
		$post_data 	= $this->get_post_data();

		if ( ! $this->is_status_active() ) {
			// Activate license

			if ( $license_key = $this->get_field_value( 'license_key', $fields['license_key'], $post_data ) ) {

				$response = WC_Plugin_API_Wrapper::activate_license( $license_key );

				if ( ! is_wp_error( $response ) && isset( $response->api_key ) ) {

					$this->settings[ 'license_key' ] = $license_key;
					$this->settings[ 'api_key' ] 	 = $response->api_key;

					$this->set_status_active();

					set_transient( $this->get_transient_status_key(), $this->get_status(), 2 * HOUR_IN_SECONDS );

				} else {
					$this->add_error( $response->get_error_message() );
				}
			}

		} elseif ( $this->is_status_active() and ! empty( $post_data['save'] ) && 'deactivate' === $post_data['save'] ) {
			// Deactivation license

			$response = WC_Plugin_API_Wrapper::deactivate_license( $this->license_key, $this->api_key );

			if ( ! is_wp_error( $response ) ) {

				$this->settings[ 'license_key' ] = '';
				$this->settings[ 'api_key' ] 	 = '';
				$this->set_status_no_active();

			} else {
				$this->add_error( $response->get_error_message() );
			}

			delete_transient( $this->get_transient_status_key() );

		}

		return update_option( $this->get_option_key(), $this->settings );
	}

	/**
	 * Clear options
	 *
	 * @return void
	 */
	public function clear_admin_options() {

		$this->settings[ 'license_key' ] = '';
		$this->settings[ 'api_key' ] 	 = '';
		$this->set_status_no_active();

		delete_transient( $this->get_transient_status_key() );

		update_option( $this->get_option_key(), $this->settings );
	}

	/**
	 * Generate Status Info HTML.
	 *
	 * @param  mixed $key
	 * @param  mixed $data
	 * @since  1.0.0
	 * @return string
	 */
	public function generate_status_info_html( $key, $data ) {

		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title'  => ''
		);

		$data  	= wp_parse_args( $data, $defaults );

		$style 	= 'color:white; padding: 3px 6px; background:' . ('active' === $this->get_status() ? 'green' : 'red') . ';';
		$text	= sprintf( __( 'you are %s receiving updates', 'wc-price-based-country-pro' ), $this->is_status_active() ? '' : sprintf( __( '%snot%s', 'wc-price-based-country-pro' ), '<strong>', '</strong>' ) );

		ob_start();
		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
					<span style="color:white; padding: 3px 6px; background:<?php echo ( $this->is_status_active() ? 'green' : 'red'); ?>">
						<?php echo ucwords( $this->get_status() ); ?>
					</span>
					&nbsp; - &nbsp; <?php echo $text; ?>
				</fieldset>
			</td>
		</tr>
		<?php

		return ob_get_clean();
	}

	/**
	 * Generate Toggle Activation HTML.
	 *
	 * @param  mixed $key
	 * @param  mixed $data
	 * @since  1.0.0
	 * @return string
	 */
	public function generate_toggle_activation_html( $key, $data ) {
		$field_key = $this->get_field_key( $key );
		$defaults  = array(
			'title' => ''
		);

		$data = wp_parse_args( $data, $defaults );

		ob_start();

		?>
		<tr valign="top">
			<th scope="row" class="titledesc">
				<label for="<?php echo esc_attr( $field_key ); ?>"><?php echo wp_kses_post( $data['title'] ); ?></label>
			</th>
			<td class="forminp">
				<fieldset>
				<?php
					if ( $this->is_status_active() ){
						echo '<button type="submit" name="save" value="deactivate">' .  __( 'Deactivate License', 'wc-price-based-country-pro' ) . '</button>';
						echo '<p class="description">' .  __( 'Deactivate your license so you can activate it  on another WooCommerce site', 'wc-price-based-country-pro' ) . '</p>';
					} else {
						_e( 'First add your Price Based on Country license key.', 'wc-price-based-country-pro' );
					}
				?>
				</fieldset>
			</td>
		</tr>
		<?php

		return ob_get_clean();
	}

	/**
	 * Generate License key input box
	 *
	 * @param  mixed $key
	 * @param  mixed $data
	 * @since  1.0.0
	 * @return string
	 */
	public function generate_license_key_html( $key, $data ) {

		if ( $this->is_status_active() ) {
			$data['disabled'] = true;
			$this->settings[ $key ] = str_repeat( '*', 24 ) . substr( $this->settings[ $key ], -6 );
		}

		return $this->generate_text_html( $key, $data );
	}


	/**
	 * Singelton implementation
	 *
	 * @return WCPBC_License_Settings
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}

	/**
	 * Output option fields
	 */
	public static function output_fields(){
		self::instance()->admin_options();
	}

	/**
	 * Save option fields
	 */
	public static function save_fields(){
		self::instance()->process_admin_options();
	}
}

endif;