<?php
/**
 * Integration with WooCommerce Subscriptions by Prospress.
 *
 * @package WCPBC
 * @version  2.5.4
 */

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'WCPBC_Subscriptions' ) ) {
	exit;
}

/**
 * WCPBC_Subscriptions class.
 */
class WCPBC_Subscriptions {

	/**
	 * Hook actions and filters
	 *
	 * @since 1.0
	 */
	public static function init() {

		add_filter( 'wc_price_based_country_product_types_overriden', array( __CLASS__, 'product_types_overriden' ) );
		add_filter( 'wc_price_based_country_price_meta_keys', array( __CLASS__, 'price_meta_keys' ) );
		add_filter( 'wc_price_based_country_parent_product_types', array( __CLASS__, 'parent_product_types' ) );
		add_action( 'wc_price_based_country_frontend_princing_init', array( __CLASS__, 'frontend_init' ) );
		add_filter( 'woocommerce_subscriptions_product_price_string', array( __CLASS__, 'product_price_string' ), 10, 2 );
		add_filter( 'woocommerce_get_cart_item_from_session', array( __CLASS__, 'get_cart_item_from_session' ), 100, 3 );
		add_action( 'woocommerce_checkout_create_order', array( __CLASS__, 'checkout_create_order' ), 1000 );

		if ( is_admin() || ( defined( 'WP_CLI' ) && WP_CLI ) ) {
			// Admin.
			add_filter( 'wc_price_based_country_product_simple_fields', array( __CLASS__, 'product_simple_fields' ) );
			add_filter( 'wc_price_based_country_product_variation_fields', array( __CLASS__, 'product_variation_fields' ), 10, 2 );
			add_action( 'woocommerce_process_product_meta_subscription', array( __CLASS__, 'process_product_meta' ), 20 );
			add_action( 'woocommerce_save_product_variation', array( __CLASS__, 'process_variation_meta' ), 5, 2 );
			add_action( 'woocommerce_product_bulk_edit_save', array( __CLASS__, 'product_bulk_edit_save' ), 30 );
			add_action( 'woocommerce_bulk_edit_variations', array( __CLASS__, 'bulk_edit_variations' ), 30, 4 );
			add_action( 'wc_price_based_country_after_bulk_edit_variation', array( __CLASS__, 'after_bulk_edit_variation' ), 10, 3 );

			include_once dirname( __FILE__ ) . '/includes/class-wcpbc-subscription-reports.php';
		}

	}

	/**
	 * Add subscription product types to the properties. WC 3.6 compatibility.
	 *
	 * @param array $types Array of product types.
	 */
	public static function product_types_overriden( $types ) {
		array_push( $types, 'variable-subscription', 'subscription' );
		return $types;
	}

	/**
	 * Add product subscriptions price meta keys
	 *
	 * @param array $meta_keys Array of price meta keys.
	 * @return array
	 */
	public static function price_meta_keys( $meta_keys ) {
		array_push( $meta_keys, '_subscription_price', '_subscription_sign_up_fee' );
		return $meta_keys;
	}

	/**
	 * Add variable subscriptions product type
	 *
	 * @param array $types Array of parent product types.
	 * @return array
	 */
	public static function parent_product_types( $types ) {
		array_push( $types, 'variable-subscription' );
		return $types;
	}

	/**
	 * Frontend init hooks
	 *
	 * @since 1.0
	 */
	public static function frontend_init() {
		add_filter( 'woocommerce_product_class', array( __CLASS__, 'overwrite_subscription_classes' ), 110 );
		add_filter( 'woocommerce_product_get__subscription_sign_up_fee', array( __CLASS__, 'subscription_sign_up_fee' ), 10, 2 );
		add_filter( 'woocommerce_product_get__subscription_price', array( __CLASS__, 'subscription_price' ), 10, 2 );
		add_filter( 'woocommerce_product_get__min_price_variation_id', array( __CLASS__, 'min_price_variation_id' ), 10, 2 );
		add_filter( 'woocommerce_product_variation_get__subscription_sign_up_fee', array( __CLASS__, 'subscription_sign_up_fee' ), 10, 2 );
		add_filter( 'woocommerce_product_variation_get__subscription_price', array( __CLASS__, 'variation_subscription_price' ), 10, 2 );
	}

	/**
	 * Overwrite WC_Product_Subscription class.
	 *
	 * @since 1.0
	 * @param string $classname Class name.
	 */
	public static function overwrite_subscription_classes( $classname ) {

		if ( 'WC_Product_Subscription_Legacy' === $classname ) {
			require_once 'includes/class-wcpbc-product-subscription.php';
			$classname = 'WCPBC_Product_Subscription';
		} elseif ( 'WC_Product_Variable_Subscription_Legacy' === $classname ) {
			require_once 'includes/class-wcpbc-product-variable-subscription.php';
			$classname = 'WCPBC_Product_Variable_Subscription';
		} elseif ( 'WC_Product_Subscription' === $classname && version_compare( WC_VERSION, '3.0', '<' ) ) {
			require_once 'includes/legacy/class-wcpbc-product-subscription-legacy.php';
			$classname = 'WCPBC_Product_Subscription_Legacy';
		} elseif ( 'WC_Product_Variable_Subscription' === $classname && version_compare( WC_VERSION, '3.0', '<' ) ) {
			require_once 'includes/legacy/class-wcpbc-product-variable-subscription-legacy.php';
			$classname = 'WCPBC_Product_Variable_Subscription_Legacy';
		}

		return $classname;
	}

	/**
	 * Return subscription sign up fee
	 *
	 * @since 1.1.3
	 * @param string  $value Sign up fee value.
	 * @param WC_Data $data  WC_Data object.
	 */
	public static function subscription_sign_up_fee( $value, $data ) {
		if ( 'variable-subscription' === $data->get_type() ) {
			$min_variation_id = WCPBC_Frontend_Pricing::get_price_metadata( false, $data->get_id(), '_min_subscription_sign_up_fee_variation_id', true );
			if ( $min_variation_id ) {
				$value = WCPBC_Frontend_Pricing::get_price_metadata( $value, $min_variation_id, '_subscription_sign_up_fee', true );
			}
		} else {
			$value = WCPBC_Frontend_Pricing::get_price_metadata( $value, $data->get_id(), '_subscription_sign_up_fee', true );
		}
		return $value;
	}

	/**
	 * Return subscription price for variable subscription products
	 *
	 * @since 1.1.3
	 * @param string  $value Subscription price value.
	 * @param WC_Data $data  WC_Data object.
	 */
	public static function subscription_price( $value, $data ) {
		return WCPBC_Frontend_Pricing::get_price_metadata( $value, $data->get_id(), '_price', true );
	}

	/**
	 * Return subscription min price variation ID for variable subscription products
	 *
	 * @since 1.1.4
	 * @param string  $value Min variation ID.
	 * @param WC_Data $data  WC_Data object.
	 */
	public static function min_price_variation_id( $value, $data ) {
		$min_price_variation_id = WCPBC_Frontend_Pricing::get_price_metadata( $value, $data->get_id(), '_min_price_variation_id', true );
		$value                  = empty( $min_price_variation_id ) ? $value : $min_price_variation_id;
		return $value;
	}

	/**
	 * Return subscription price for variation subscription product
	 *
	 * @since 1.1.4
	 * @param string  $value Variation subscription price.
	 * @param WC_Data $data  WC_Data object.
	 */
	public static function variation_subscription_price( $value, $data ) {
		return WCPBC_Frontend_Pricing::get_price_metadata( $value, $data->get_id(), '_subscription_price', true );
	}

	/**
	 * Add a wrapper for ajax geolocation
	 *
	 * @since 2.2.7
	 * @param string     $subscription_string Subscription price string.
	 * @param WC_Product $product Product instance.
	 */
	public static function product_price_string( $subscription_string, $product ) {

		if ( is_callable( array( 'WCPBC_Ajax_Geolocation', 'is_enabled' ) ) && WCPBC_Ajax_Geolocation::is_enabled() ) {
			$subscription_string = WCPBC_Ajax_Geolocation::wrapper_price( $product, $subscription_string );
		}

		return $subscription_string;
	}

	/**
	 * Apply the exchage rates to the renewal order product object.
	 *
	 * @since 2.3.3
	 * @param array  $cart_item_session_data Session data.
	 * @param array  $cart_item              Item values.
	 * @param string $key                   Current session data key.
	 * @return array
	 */
	public static function get_cart_item_from_session( $cart_item_session_data, $cart_item, $key ) {

		if ( ! empty( $cart_item_session_data['subscription_renewal']['renewal_order_id'] ) ) {

			$order = wc_get_order( $cart_item_session_data['subscription_renewal']['renewal_order_id'] );

			if ( $order ) {

				$order_currency   = $order->get_currency();
				$current_currency = WCPBC()->current_zone ? WCPBC()->current_zone->get_currency() : wcpbc_get_base_currency();

				if ( $order_currency !== $current_currency ) {

					$_product   = $cart_item_session_data['data'];
					$price      = $_product->get_price();
					$order_zone = WCPBC_Pricing_Zones::get_zone_from_order( $order );

					if ( $order_zone ) {
						if ( $order_zone->get_currency() !== $order_currency ) {

							$order_zone = new WCPBC_Pricing_Zone(
								array(
									'exchange_rate' => WCPBC_Update_Exchange_Rates::get_exchange_rate_from_api( $order_currency ),
									'currency'      => $order_currency,
								)
							);
						}

						$price = $order_zone->get_base_currency_amount( $price );
					}

					if ( WCPBC()->current_zone ) {
						$price = WCPBC()->current_zone->get_exchange_rate_price( $price );
					}

					// Set the product price.
					$_product->set_price( $price );
				}
			}
		}

		return $cart_item_session_data;
	}

	/**
	 * Fix order currency. WooCommerce Subscription copy metadata from the original subscription and overwrite the WooCommerce currency.
	 *
	 * @see WCS_Cart_Early_Renewal::copy_subscription_meta_to_order
	 * @param WC_Order $order The WC Order object.
	 *
	 * @since 2.6.3
	 */
	public static function checkout_create_order( $order ) {
		if ( $order->get_currency() !== get_woocommerce_currency() && function_exists( 'wcs_cart_contains_renewal' ) && wcs_cart_contains_renewal() ) {
			$order->set_currency( get_woocommerce_currency() );
			$order->calculate_totals();
		}
	}

	/**
	 * Add the subscription fields to product simple.
	 *
	 * @since 2.5
	 * @param array $fields Product simple fields.
	 * @return array
	 */
	public static function product_simple_fields( $fields ) {

		$fields[] = array(
			'name'              => '_subscription_price',
			'class'             => 'short wc_input_price wcpbc_input_subscription_price',
			'wrapper_class'     => 'wcpbc_show_if_manual_subscription',
			// translators: %s is a currency symbol.
			'label'             => __( 'Subscription Price (%s)', 'woocommerce-subscriptions' ),
			'placeholder'       => _x( 'e.g. 5.90', 'example price', 'woocommerce-subscriptions' ),
			'type'              => 'text',
			'custom_attributes' => array(
				'step' => 'any',
				'min'  => '0',
			),
		);

		$fields[] = array(
			'name'              => '_subscription_sign_up_fee',
			'class'             => 'short wc_input_price',
			'wrapper_class'     => 'wcpbc_show_if_manual_subscription',
			// translators: %s is a currency symbol.
			'label'             => __( 'Sign-up Fee (%s)', 'woocommerce-subscriptions' ),
			'placeholder'       => _x( 'e.g. 9.90', 'example price', 'woocommerce-subscriptions' ),
			'description'       => __( 'Optionally include an amount to be charged at the outset of the subscription. The sign-up fee will be charged immediately, even if the product has a free trial or the payment dates are synced.', 'woocommerce-subscriptions' ),
			'desc_tip'          => true,
			'type'              => 'text',
			'custom_attributes' => array(
				'step' => 'any',
				'min'  => '0',
			),
		);

		return $fields;
	}

	/**
	 * Add the subscription fields to product variation.
	 *
	 * @since 2.5
	 * @param array $fields Product simple fields.
	 * @param int   $loop Index of loop variation.
	 * @return array
	 */
	public static function product_variation_fields( $fields, $loop ) {
		$fields['_subscription_sign_up_fee'] = array(
			'name'          => "_variable_subscription_sign_up_fee[$loop]",
			// Translators: currency symbol.
			'label'         => __( 'Sign-up Fee (%s)', 'woocommerce-subscriptions' ),
			'wrapper_class' => 'form-row form-row-first wcpbc_show_if_manual_subscription',
		);
		$fields['_subscription_price']       = array(
			'name'          => "_variable_subscription_price[$loop]",
			'class'         => 'wcpbc_input_subscription_price',
			// Translators: currency symbol.
			'label'         => __( 'Subscription Price (%s)', 'woocommerce-subscriptions' ),
			'wrapper_class' => 'form-row form-row-last wcpbc_show_if_manual_subscription',
		);

		return $fields;
	}

	/**
	 * Save product metadata
	 *
	 * @param int $post_id Post ID.
	 */
	public static function process_product_meta( $post_id ) {
		$postdata = array(
			'subscription_price'       => isset( $_POST['_subscription_price'] ) ? wc_clean( wp_unslash( $_POST['_subscription_price'] ) ) : '', // WPCS: CSRF ok.
			'subscription_sign_up_fee' => isset( $_POST['_subscription_sign_up_fee'] ) ? wc_clean( wp_unslash( $_POST['_subscription_sign_up_fee'] ) ) : '', // WPCS: CSRF ok.
		);
		self::save_subscription_metadata( $post_id, $postdata );
		WCPBC_Admin_Meta_Boxes::process_product_meta( $post_id );
	}

	/**
	 * Save product metadata
	 *
	 * @param int $post_id Post ID.
	 * @param int $index Index of variations to save.
	 */
	public static function process_variation_meta( $post_id, $index ) {
		$product_type = isset( $_POST['product-type'] ) ? wc_clean( wp_unslash( $_POST['product-type'] ) ) : false; // WPCS: CSRF ok.
		if ( 'variable-subscription' !== $product_type ) {
			return;
		}
		$postdata = array(
			'subscription_price'       => isset( $_POST['variable_subscription_price'][ $index ] ) ? wc_clean( wp_unslash( $_POST['variable_subscription_price'][ $index ] ) ) : '', // WPCS: CSRF ok.
			'subscription_sign_up_fee' => isset( $_POST['variable_subscription_sign_up_fee'][ $index ] ) ? wc_clean( wp_unslash( $_POST['variable_subscription_sign_up_fee'][ $index ] ) ) : '', // WPCS: CSRF ok.
		);
		self::save_subscription_metadata( $post_id, $postdata, $index );
	}

	/**
	 * Save subscription metada.
	 *
	 * @param int   $post_id Post ID.
	 * @param array $postdata Array with the subscription_price and subscription_sign_up_fee values.
	 * @param int   $index Index of variations to save.
	 */
	private static function save_subscription_metadata( $post_id, $postdata, $index = false ) {
		$variable          = false === $index ? '' : '_variable';
		$regular_price_key = $variable . '_regular_price';

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {

			if ( wcpbc_is_exchange_rate( $zone->get_input_var( $variable . '_price_method', $index ) ) ) {

				$_subscription_price       = $zone->get_exchange_rate_price( $postdata['subscription_price'], false );
				$_subscription_sign_up_fee = $zone->get_exchange_rate_price( $postdata['subscription_sign_up_fee'], false );

			} else {

				$_subscription_price       = $zone->get_input_var( $variable . '_subscription_price', $index );
				$_subscription_sign_up_fee = $zone->get_input_var( $variable . '_subscription_sign_up_fee', $index );

				// Copy the subscription price to regular price.
				$_regular_price_key = $zone->get_postmetakey( $regular_price_key );
				if ( false === $index ) {
					$_POST[ $_regular_price_key ] = $_subscription_price;
				} else {
					$_POST[ $_regular_price_key ][ $index ] = $_subscription_price;
				}

				// Sanitize values.
				$_subscription_price       = wc_format_decimal( $_subscription_price );
				$_subscription_sign_up_fee = wc_format_decimal( $_subscription_sign_up_fee );
			}

			// Save metadata.
			$zone->set_postmeta( $post_id, '_subscription_price', $_subscription_price );
			$zone->set_postmeta( $post_id, '_subscription_sign_up_fee', $_subscription_sign_up_fee );
		}
	}

	/**
	 * Quick and Bulk product edit.
	 *
	 * @param WC_Product $product Product instance.
	 */
	public static function product_bulk_edit_save( $product ) {

		$_product = (object) wcpbc_get_prop_value( $product, array( 'id', 'type' ) );

		if ( 'subscription' !== $_product->type ) {
			return;
		}

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			if ( $zone->is_exchange_rate_price( $_product->id ) ) {
				$_subscription_price       = $zone->get_exchange_rate_price_by_post( $_product->id, '_subscription_price', false );
				$_subscription_sign_up_fee = $zone->get_exchange_rate_price_by_post( $_product->id, '_subscription_sign_up_fee', false );

				$zone->set_postmeta( $_product->id, '_subscription_price', $_subscription_price );
				$zone->set_postmeta( $_product->id, '_subscription_sign_up_fee', $_subscription_sign_up_fee );
			}
		}
	}

	/**
	 * Bulk edit variations via AJAX.
	 *
	 * @param string $bulk_action Variation bulk action.
	 * @param array  $data Sanitized post data.
	 * @param int    $product_id Variable product ID.
	 * @param array  $variations Array of varations ID.
	 */
	public static function bulk_edit_variations( $bulk_action, $data, $product_id, $variations ) {
		$actions      = array( 'variable_subscription_sign_up_fee', 'variable_regular_price', 'variable_regular_price_increase', 'variable_regular_price_decrease' );
		$product_type = isset( $_POST['product_type'] ) ? wc_clean( wp_unslash( $_POST['product_type'] ) ) : ''; // WPCS: CSRF ok.

		if ( ! ( in_array( $bulk_action, $actions, true ) && 'variable-subscription' === $product_type ) ) {
			return;
		}

		foreach ( WCPBC_Pricing_Zones::get_zones() as $zone ) {
			foreach ( $variations as $variation_id ) {
				if ( $zone->is_exchange_rate_price( $variation_id ) ) {
					if ( 'variable_subscription_sign_up_fee' === $bulk_action ) {
						$_subscription_sign_up_fee = $zone->get_exchange_rate_price_by_post( $variation_id, '_subscription_sign_up_fee', false );
						$zone->set_postmeta( $variation_id, '_subscription_sign_up_fee', $_subscription_sign_up_fee );
					} else {
						// Copy the regular price to the subscription price.
						$_subscription_price = $zone->get_postmeta( $variation_id, '_regular_price', false );
						$zone->set_postmeta( $variation_id, '_subscription_price', $_subscription_price );
					}
				}
			}
		}
	}

	/**
	 * After bulk editing variation pricing zone.
	 *
	 * @param int                $variation_id Variable product ID.
	 * @param WCPBC_Pricing_zone $zone Array of varations ID.
	 * @param string             $field Field edited.
	 */
	public static function after_bulk_edit_variation( $variation_id, $zone, $field ) {
		if ( '_regular_price' !== $field ) {
			return;
		}
		// Copy the regular price to the subscription price.
		$_subscription_price = $zone->get_postmeta( $variation_id, '_regular_price', false );
		$zone->set_postmeta( $variation_id, '_subscription_price', $_subscription_price );
	}
}

WCPBC_Subscriptions::init();

