<?php
/**
 * Rest API Logs
 *
 * @package     GamiPress\Rest_API\Logs
 * @author      GamiPress <contact@gamipress.com>, Ruben Garcia <rubengcdev@gmail.com>
 * @since       1.6.5
 */
// Exit if accessed directly
if( !defined( 'ABSPATH' ) ) exit;

/**
 * User Earning item schema
 *
 * @since 1.6.5
 *
 * @param array $schema
 *
 * @return array
 */
function gamipress_user_earnings_rest_item_schema( $schema ) {

    // Properties
    $schema['properties'] = array_merge( $schema['properties'], array(
        'title' => array(
            'description' => __( 'The title for the object.', 'gamipress' ),
            'type'        => 'string',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'user_id' => array(
            'description' => __( 'The ID for the user of the object.', 'gamipress' ),
            'type'        => 'integer',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'post_id' => array(
            'description' => __( 'The ID for the post of the object.', 'gamipress' ),
            'type'        => 'integer',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'post_type' => array(
            'description' => __( 'The type for the post of the object.', 'gamipress' ),
            'type'        => 'string',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'points' => array(
            'description' => __( 'The points amount of the object.', 'gamipress' ),
            'type'        => 'integer',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'points_type' => array(
            'description' => __( 'The type for the points amount of the object.', 'gamipress' ),
            'type'        => 'string',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
        'date'            => array(
            'description' => __( 'The date the object was created, in the site\'s timezone.', 'gamipress' ),
            'type'        => 'string',
            'format'      => 'date-time',
            'context'     => array( 'view', 'edit', 'embed' ),
        ),
    ) );

    return $schema;

}
add_filter( 'ct_rest_gamipress_user_earnings_schema', 'gamipress_user_earnings_rest_item_schema' );

/**
 * User Earnings collection params
 *
 * @since 1.6.5
 *
 * @param array     $query_params
 * @param CT_Table  $ct_table
 *
 * @return array
 */
function gamipress_user_earnings_rest_collection_params( $query_params, $ct_table ) {

    // User ID
    $query_params['user_id'] = array(
        'description'           => __( 'Limit result set to earnings assigned to specific users.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'integer',
        ),
        'default'               => array(),
    );

    // Post ID
    $query_params['post_id'] = array(
        'description'           => __( 'Limit result set to earnings assigned to specific posts.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'integer',
        ),
        'default'               => array(),
    );

    // Post Type
    $query_params['post_type'] = array(
        'description'           => __( 'Limit result set to earnings assigned to specific post types.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'string',
        ),
        'default'               => array(),
    );

    // Points Type
    $query_params['points_type'] = array(
        'description'           => __( 'Limit result set to earnings assigned to specific points types.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'string',
        ),
        'default'               => array(),
    );

    // Force Types
    $query_params['force_types'] = array(
        'description'           => __( 'Setting it to true will force to limit results of the same post_type and points_type.' ),
        'type'                  => 'boolean',
        'default'               => false,
    );

    // Exclude ( exclude => user_earning__not_in )
    $query_params['exclude'] = array(
        'description'           => __( 'Ensure result set excludes specific IDs.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'integer',
        ),
        'default'               => array(),
    );

     // Include ( include => user_earning__in )
    $query_params['include'] = array(
        'description'           => __( 'Limit result set to specific IDs.', 'gamipress' ),
        'type'                  => 'array',
        'items'                 => array(
            'type'              => 'integer',
        ),
        'default'               => array(),
    );

    // After
    $query_params['after'] = array(
        'description'           => __( 'Limit response to user earnings after a given ISO8601 compliant date.', 'gamipress' ),
        'type'                  => 'string',
        'format'                => 'date-time',
    );

    // Before
    $query_params['before'] = array(
        'description'           => __( 'Limit response to user earnings before a given ISO8601 compliant date.', 'gamipress' ),
        'type'                  => 'string',
        'format'                => 'date-time',
    );


    return $query_params;
}
add_filter( 'ct_rest_gamipress_user_earnings_collection_params', 'gamipress_user_earnings_rest_collection_params', 10, 2 );

/**
 * Set custom parameters mapping
 *
 * @since 1.6.5
 *
 * @param array             $parameter_mappings Array of parameters to map.
 * @param CT_Table          $ct_table           Table object.
 * @param WP_REST_Request   $request            The request given.
 *
 * @return array
 */
function gamipress_user_earnings_rest_parameter_mappings( $parameter_mappings, $ct_table, $request ) {

    $parameter_mappings['exclude'] = 'user_earning__not_in';
    $parameter_mappings['include'] = 'user_earning__in';

    return $parameter_mappings;
}
add_filter( 'ct_rest_gamipress_user_earnings_parameter_mappings', 'gamipress_user_earnings_rest_parameter_mappings', 10, 3 );

/**
 * Fields sanitization
 *
 * @since 1.6.5
 *
 * @param mixed             $value      The field value given.
 * @param string            $field      The field name.
 * @param WP_REST_Request   $request    The request object.
 *
 * @return mixed|WP_Error
 */
function gamipress_user_earnings_rest_sanitize_field_value( $value, $field, $request ) {

    switch( $field ) {
        case 'user_id':
            // Validate user ID
            $value = absint( $value );
            $user_obj = get_userdata( $value );

            if ( ! $user_obj ) {
                return new WP_Error( 'rest_invalid_field', __( 'Invalid user ID.', 'gamipress' ), array( 'status' => 400 ) );
            }
            break;
        case 'post_id':
            // Validate post ID
            $value = absint( $value );
            $post_obj = get_post( $value );

            if ( ! $post_obj ) {
                return new WP_Error( 'rest_invalid_field', __( 'Invalid post ID.', 'gamipress' ), array( 'status' => 400 ) );
            }
            break;
        case 'points_type':
            // Validate points type
            if( ! empty( $value ) ) {
                if ( ! in_array( $value, gamipress_get_points_types_slugs() ) ) {
                    return new WP_Error( 'rest_invalid_field', __( 'Invalid points type.', 'gamipress' ), array( 'status' => 400 ) );
                }
            }
            break;
        case 'date':
            // Validate date.
            $mm = substr( $value, 5, 2 );
            $jj = substr( $value, 8, 2 );
            $aa = substr( $value, 0, 4 );
            $valid_date = wp_checkdate( $mm, $jj, $aa, $value );

            if ( ! $valid_date ) {
                return new WP_Error( 'rest_invalid_field', __( 'Invalid date.', 'gamipress' ), array( 'status' => 400 ) );
            }
            break;
    }

    return $value;
}
add_filter( 'ct_rest_gamipress_user_earnings_sanitize_field_value', 'gamipress_user_earnings_rest_sanitize_field_value', 10, 3 );